const script = registerScript({
    name: "RemoteControlBridge",
    version: "1.0.0",
    authors: ["Dylanvip2024"]
});

// Import essential Java classes for networking, threading, and I/O
const Thread = Java.type("java.lang.Thread");
const Timer = Java.type("java.util.Timer");
const TimerTask = Java.type("java.util.TimerTask");
const ServerSocket = Java.type("java.net.ServerSocket");
const Socket = Java.type("java.net.Socket");
const BufferedReader = Java.type("java.io.BufferedReader");
const InputStreamReader = Java.type("java.io.InputStreamReader");
const PrintWriter = Java.type("java.io.PrintWriter");
const OutputStreamWriter = Java.type("java.io.OutputStreamWriter");
const StandardCharsets = Java.type("java.nio.charset.StandardCharsets");
const LinkedBlockingQueue = Java.type("java.util.concurrent.LinkedBlockingQueue");

// Import Minecraft's packet class for detecting chat messages
const GameMessageS2CPacket = Java.type("net.minecraft.network.packet.s2c.play.GameMessageS2CPacket");

// Global state variables
let serverSocket = null;                // The server socket listening for incoming connections
let serverThread = null;                // Thread running the TCP server
let clientConnection = null;            // Current connected client socket (only one client supported)
let out = null;                         // PrintWriter for sending responses to the client
let pendingWaitRequest = null;          // Holds the active "waitForChat" request (if any)
const requestQueue = new LinkedBlockingQueue(); // Thread-safe queue for incoming client commands
const DEBUG_MODE = false;               // Enable to log debug messages in chat

/**
 * Utility function to log debug messages to Minecraft chat (only if DEBUG_MODE is true).
 * Prefixes messages with [RemoteBridge] for clarity.
 */
function logDebug(...args) {
    if (DEBUG_MODE) {
        const message = args.join(" ");
        Client.displayChatMessage(`§7[RemoteBridge] §f${message}`);
    }
}

// Register the main module
script.registerModule({
    name: "RemoteControlBridge",
    category: "Misc",
    description: "Provides a TCP interface to send chat commands and wait for new chat messages from the game." // 
}, (module) => {
    const PORT = 25560;        // Default port for the TCP server
    let isRunning = false;     // Indicates whether the server is currently active

    /**
     * Starts the TCP server on the specified port.
     * Accepts only one client connection at a time.
     * Spawns a background thread to handle client I/O.
     */
    const startServer = () => {
        try {
            serverSocket = new ServerSocket(PORT);
            isRunning = true;
            Client.displayChatMessage(`§a[RemoteControlBridge] Listening on port: ${PORT}`);
            logDebug("Server started successfully");

            // Server thread: continuously accepts new client connections
            serverThread = new Thread(() => {
                while (isRunning && serverSocket && !serverSocket.isClosed()) {
                    try {
                        const socket = serverSocket.accept();

                        // Only allow one client at a time
                        if (clientConnection) {
                            socket.close();
                            logDebug("Rejected duplicate connection attempt");
                            continue;
                        }

                        // Register the new client
                        clientConnection = socket;
                        out = new PrintWriter(
                            new OutputStreamWriter(socket.getOutputStream(), StandardCharsets.UTF_8),
                            true  // Auto-flush
                        );
                        logDebug("Client connected");

                        // Spawn a dedicated thread to read commands from this client
                        new Thread(() => {
                            try {
                                const reader = new BufferedReader(
                                    new InputStreamReader(socket.getInputStream(), StandardCharsets.UTF_8)
                                );
                                let line;
                                // Read each line (JSON command) until connection closes
                                while (isRunning && (line = reader.readLine()) !== null) {
                                    logDebug("Received request:", line);
                                    requestQueue.offer(line); // Enqueue for processing on main thread
                                }
                            } catch (e) {
                                if (isRunning) {
                                    logDebug("Error reading from client:", e.message);
                                }
                            } finally {
                                cleanupClient(); // Ensure resources are released
                            }
                        }).start();

                        // Notify client that connection was accepted
                        sendResponse({ type: "connected", port: PORT });
                    } catch (e) {
                        // Only log errors if the server is still supposed to be running
                        if (isRunning && serverSocket && !serverSocket.isClosed()) {
                            logDebug("Server error:", e.message);
                        }
                    }
                }
            });

            serverThread.setDaemon(true); // Daemon thread won't prevent JVM shutdown
            serverThread.start();
        } catch (e) {
            Client.displayChatMessage(`§c[RemoteControlBridge] Failed to start: ${e.message}`);
        }
    };

    /**
     * Gracefully closes the current client connection and cleans up associated resources.
     */
    const cleanupClient = () => {
        try {
            if (out) out.close();
            if (clientConnection) clientConnection.close();
        } catch (ignored) {}
        out = null;
        clientConnection = null;
        logDebug("Client connection closed");
    };

    /**
     * Sends a JSON response back to the connected client.
     * @param {Object} obj - The response object to serialize and send.
     */
    const sendResponse = (obj) => {
        if (out && !out.checkError()) {
            const jsonStr = JSON.stringify(obj);
            out.println(jsonStr);
            logDebug("Sent response:", jsonStr);
        }
    };

    /**
     * Processes a single JSON command received from the client.
     * Supports: sendChat, waitForChat, ping.
     */
    const handleRequest = (jsonStr) => {
        try {
            const req = JSON.parse(jsonStr);
            const cmd = req.cmd;
            logDebug("Handling command:", cmd);

            // Ensure the player is in a world
            if (!mc.player) {
                return sendResponse({ error: "Player is not in a world" });
            }

            switch (cmd) {
                case "sendChat":
                    // Validate message field
                    if (!req.message || typeof req.message !== 'string') {
                        return sendResponse({ error: "Invalid message content" });
                    }
                    // Send as command if starts with '/', otherwise regular chat
                    if (req.message.startsWith('/')) {
                        NetworkUtil.sendCommand(req.message.substring(1));
                    } else {
                        NetworkUtil.sendChatMessage(req.message);
                    }
                    sendResponse({ type: "action", result: "messageSent", message: req.message });
                    break;

                case "waitForChat":
                    // Only allow one pending waitForChat request at a time
                    if (pendingWaitRequest) {
                        return sendResponse({ error: "Another waitForChat request is already pending" });
                    }

                    // Enforce reasonable timeout bounds (default: 30s, max: 60s)
                    const timeoutMs = Math.min(req.timeout || 30000, 60000);
                    const timer = new Timer();

                    // Set up a timeout task to cancel the wait if no message arrives
                    const timeoutTask = new TimerTask({
                        run: () => {
                            if (pendingWaitRequest) {
                                sendResponse({ type: "chat", result: "timeout" });
                                pendingWaitRequest = null;
                                logDebug("waitForChat timed out");
                            }
                        }
                    });

                    timer.schedule(timeoutTask, timeoutMs);
                    pendingWaitRequest = { resolve: sendResponse, timer: timer };
                    logDebug("Waiting for chat message (timeout:", timeoutMs + "ms)");
                    break;

                case "ping":
                    // Simple heartbeat to check if the bridge is responsive
                    sendResponse({ type: "pong" });
                    break;

                default:
                    sendResponse({ error: `Unknown command: ${cmd}` });
            }
        } catch (e) {
            sendResponse({ error: `Processing error: ${e.message}` });
        }
    };

    // Process queued requests every tick (ensures safe interaction with Minecraft's main thread)
    module.on("playerTick", () => {
        if (!isRunning) return;
        let req;
        while ((req = requestQueue.poll()) !== null) {
            handleRequest(req);
        }
    });

    // Listen for incoming chat messages from the server
    module.on("packet", (event) => {
        if (!isRunning || !pendingWaitRequest) return;

        const packet = event.packet;

        // Only process genuine chat messages (not system or action bar)
        if (packet instanceof GameMessageS2CPacket) {
            try {
                // Extract plain-text message content
                const messageText = packet.content().getString();
                logDebug("Received chat message:", messageText);

                // Ignore empty or whitespace-only messages
                if (messageText && messageText.trim().length > 0) {
                    const { resolve, timer } = pendingWaitRequest;
                    logDebug("Forwarding message to client:", messageText);
                    resolve({ type: "chat", result: "newMessage", message: messageText });
                    timer.cancel(); // Stop timeout timer
                    pendingWaitRequest = null; // Clear pending request
                }
            } catch (e) {
                logDebug("Error processing chat message:", e.message);
            }
        }
    });

    // Start the TCP server when the module is enabled
    module.on("enable", startServer);

    // Clean up all resources when the module is disabled
    module.on("disable", () => {
        isRunning = false;

        // Cancel any pending waitForChat request
        if (pendingWaitRequest) {
            pendingWaitRequest.timer.cancel();
            pendingWaitRequest = null;
        }

        // Interrupt and clean up threads/sockets
        if (serverThread) serverThread.interrupt();
        if (serverSocket) {
            try { serverSocket.close(); } catch (ignored) {}
        }
        cleanupClient();
        requestQueue.clear();

        Client.displayChatMessage("§c[RemoteControlBridge] Stopped");
        logDebug("Module disabled");
    });
});