import socket
import json
import sys
import time
import re


class RemoteControlClient:
    """
    A Python client for interacting with the RemoteControlBridge module in Minecraft.
    It connects via TCP to send chat messages and listen for incoming chat messages,
    with optional filtering of messages sent by the player themselves.
    """

    def __init__(self, host='127.0.0.1', port=25560, player_name=None):
        """
        Initialize the client.

        :param host: IP address of the Minecraft client (default: localhost)
        :param port: TCP port the RemoteControlBridge is listening on (default: 25560)
        :param player_name: Optional. If provided, messages originating from this player
                            will be filtered out during chat listening (e.g., to avoid echo).
        """
        self.host = host
        self.port = port
        self.sock = None          # Socket connection to the Minecraft bridge
        self.fileobj = None       # Buffered text reader for socket
        self.connected = False    # Connection status flag
        self.player_name = player_name  # Used for filtering self-sent messages

    def connect(self):
        """
        Establish a TCP connection to the RemoteControlBridge server.
        Reads the welcome message to confirm successful handshake.

        :return: True if connection succeeded, False otherwise.
        """
        try:
            self.sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            self.sock.settimeout(10)  # 10-second timeout for initial connection
            self.sock.connect((self.host, self.port))
            self.sock.settimeout(None)  # Disable timeout after connection
            self.fileobj = self.sock.makefile('r', encoding='utf-8')

            # Read the initial handshake response
            welcome_line = self.fileobj.readline().strip()
            if welcome_line:
                welcome = json.loads(welcome_line)
                if welcome.get("type") == "connected":
                    print(f"✅ Successfully connected to RemoteControlBridge ({self.host}:{self.port})")
                    if self.player_name:
                        print(f"🔍 Message filtering enabled. Messages from '{self.player_name}' will be ignored.")
                    self.connected = True
                    return True

            print("❌ Unexpected response from server during handshake.")
            return False

        except socket.timeout:
            print("❌ Connection timed out. Please verify:")
            print("   1. Minecraft is running")
            print("   2. The 'RemoteControlBridge' module is enabled in-game")
            return False
        except Exception as e:
            print(f"❌ Failed to connect: {e}")
            return False

    def send_chat_message(self, message):
        """
        Send a chat message or command to the Minecraft client.

        If the message starts with '/', it will be sent as a command.
        Otherwise, it's sent as a regular chat message.

        :param message: The message or command to send (e.g., "Hello" or "/help")
        :return: True if the message was acknowledged as sent, False otherwise.
        """
        try:
            cmd = {"cmd": "sendChat", "message": message}
            cmd_str = json.dumps(cmd) + "\n"
            self.sock.sendall(cmd_str.encode('utf-8'))

            # Wait for acknowledgment
            response_line = self.fileobj.readline().strip()
            if response_line:
                response = json.loads(response_line)
                if response.get("result") == "messageSent":
                    print(f"✅ Message sent successfully: {message}")
                    return True
                else:
                    print(f"❌ Message failed: {response.get('error', 'Unknown error')}")
                    return False
            return False

        except Exception as e:
            print(f"❌ Error while sending message: {e}")
            return False

    def is_own_message(self, message):
        """
        Determine if a chat message was sent by the player themselves.

        This uses common chat formats to match the player name, such as:
        - "PlayerName: message"
        - "<PlayerName> message"
        - "[PlayerName] message"
        - etc.

        :param message: The raw chat message string from the server.
        :return: True if the message appears to be from the local player, False otherwise.
        """
        if not self.player_name:
            return False  # Filtering disabled if no player name is set

        # Define common chat patterns where the player name appears
        patterns = [
            rf"^{re.escape(self.player_name)}:",   # "Dylan: hello"
            rf"^{re.escape(self.player_name)} >",  # "Dylan > hello"
            rf"<{re.escape(self.player_name)}>",   # "<Dylan> hello"
            rf"\[{re.escape(self.player_name)}\]", # "[Dylan] hello"
            rf"^{re.escape(self.player_name)}\s",  # "Dylan hello..." (name at start followed by space)
        ]

        message_text = message.strip()
        for pattern in patterns:
            if re.search(pattern, message_text, re.IGNORECASE):
                return True
        return False

    def wait_for_chat_message(self, timeout=60, filter_own_messages=True):
        """
        Request the server to wait for the next incoming chat message.

        This sends a "waitForChat" command and blocks until:
        - A new chat message arrives,
        - The timeout is reached, or
        - An error occurs.

        If filtering is enabled and the message matches the player's name,
        it returns a special "filtered" result instead of the message.

        :param timeout: Maximum time (in seconds) to wait for a message (max 60s enforced by server).
        :param filter_own_messages: If True, messages identified as self-sent will be filtered out.
        :return: A response dict (e.g., {"result": "newMessage", "message": "..."}) or None on error.
                 Returns {"result": "filtered"} if the message was filtered out.
        """
        try:
            cmd = {"cmd": "waitForChat", "timeout": timeout * 1000}
            cmd_str = json.dumps(cmd) + "\n"
            self.sock.sendall(cmd_str.encode('utf-8'))
            print(f"⏳ Waiting for chat message (timeout: {timeout}s)...")

            # Read exactly one response from the server
            response_line = self.fileobj.readline().strip()
            if not response_line:
                return None

            response = json.loads(response_line)

            # Check if it's a new message and whether we should filter it
            if (response.get("result") == "newMessage" and
                filter_own_messages and
                self.is_own_message(response.get("message", ""))):

                print(f"🔇 Filtered out own message: {response.get('message')}")
                return {"type": "chat", "result": "filtered"}

            return response

        except Exception as e:
            print(f"❌ Error while waiting for chat message: {e}")
            return None

    def disconnect(self):
        """Close the TCP connection and clean up resources."""
        if self.sock:
            self.sock.close()
        self.connected = False
        print("🔌 Connection closed.")


def main():
    """
    Main entry point for the demo client.
    Demonstrates:
    1. Connecting to the bridge
    2. Sending a test message
    3. Continuously listening for incoming chat messages (with self-message filtering)
    """
    print("=" * 50)
    print("RemoteControlBridge Python Client (with message filtering)")
    print("=" * 50)

    # ⚠️ Replace this with your actual Minecraft username
    PLAYER_NAME = "Dylanvip2024"

    client = RemoteControlClient(player_name=PLAYER_NAME)

    try:
        # Step 1: Connect to the Minecraft bridge
        if not client.connect():
            sys.exit(1)

        # Step 2: Send a test message
        print("\n1. Sending a test message...")
        client.send_chat_message("Hello from Python client!")

        time.sleep(2)  # Brief pause to avoid immediate echo during listening

        # Step 3: Listen for chat messages in a loop
        print("\n2. Listening for chat messages (self-messages filtered)...")
        print("💡 Press Ctrl+C to stop.")
        print("-" * 40)

        message_count = 0

        while True:
            response = client.wait_for_chat_message(timeout=60, filter_own_messages=True)

            if response is None:
                print("❌ Received empty response. Continuing...")
                continue

            if response.get("result") == "newMessage":
                message_count += 1
                message = response.get("message", "[empty]")
                print(f"💬 [{message_count}] {message}")

            elif response.get("result") == "timeout":
                print("⏱️ Timeout reached. Continuing to listen...")

            elif response.get("result") == "filtered":
                # Own message was filtered—just continue waiting
                print("🔄 Waiting for next message...")
                continue

            elif response.get("error"):
                print(f"❌ Server error: {response.get('error')}")

            else:
                print(f"📨 Unexpected response: {response}")

    except KeyboardInterrupt:
        print("\n🛑 Program interrupted by user.")
    except Exception as e:
        print(f"\n💥 Unexpected error: {e}")
    finally:
        client.disconnect()


if __name__ == "__main__":
    main()