// noinspection JSUnresolvedReference,JSVoidFunctionReturnValueUsed

const script = registerScript({
    name: "Server List Explorer Installer",
    version: "1.0.1",
    authors: ["SpoilerRules"]
});

const File = Java.type("java.io.File");
const Files = Java.type("java.nio.file.Files");
const Paths = Java.type("java.nio.file.Paths");
const StandardCopyOption = Java.type("java.nio.file.StandardCopyOption");
const SimpleFileVisitor = Java.type("java.nio.file.SimpleFileVisitor");
const FileVisitResult = Java.type("java.nio.file.FileVisitResult");
const DosFileAttributeView = Java.type("java.nio.file.attribute.DosFileAttributeView");

const Arrays = Java.type("java.util.Arrays");
const ZipFile = Java.type("java.util.zip.ZipFile");

const SystemJ = Java.type("java.lang.System");
const ProcessBuilder = Java.type("java.lang.ProcessBuilder");

const TimeUnit = Java.type("java.util.concurrent.TimeUnit");
const Thread = Java.type("java.lang.Thread");

const Okio = Java.type("okio.Okio");
const OkHttpClient = Java.type("okhttp3.OkHttpClient");
const Request = Java.type("okhttp3.Request");

const ClientChat = Java.type("net.ccbluex.liquidbounce.utils.client.ClientChat");
const MessageMetadata = Java.type("net.ccbluex.liquidbounce.utils.client.MessageMetadata");

const isWindows = () => SystemJ.getProperty("os.name").toLowerCase().contains("win");
const isMac = () => SystemJ.getProperty("os.name").toLowerCase().contains("mac");

const chat = (msg) => {
    Client.displayChatMessage(`§b[§3SLE§b] §r${msg}`);
};

const runAsync = (name, fn) => {
    Thread
        .ofVirtual()
        .name(name)
        .start(fn);
};

const javaBin = () => {
    const home = SystemJ.getProperty("java.home");
    if (!home) throw new Error("Could not determine JAVA_HOME");
    const binName = isWindows() ? "java.exe" : "java";
    return Paths.get(home, "bin", binName).toString();
};

const runDetached = (cmdArray, workDirPath) => {
    try {
        const pb = new ProcessBuilder(Arrays.asList(cmdArray));
        if (workDirPath) {
            pb.directory(new File(workDirPath.toString()));
        }
        pb.start(); // non-blocking
    } catch (e) {
        chat(`§cFailed to start process: §f${cmdArray.join(" ")} §7(${String(e)})`);
        throw e;
    }
};

const getLocalAppDataPath = () => {
    const localAppData = SystemJ.getenv("LOCALAPPDATA");
    return (localAppData && localAppData.length() > 0)
        ? Paths.get(localAppData)
        : Paths.get(SystemJ.getProperty("user.home"), "AppData", "Local");
};

const getPaths = () => {
    const userHome = SystemJ.getProperty("user.home");

    if (isWindows()) {
        const installDir = getLocalAppDataPath()
            .resolve("Programs")
            .resolve("Server List Explorer");
        return {
            installDir,
            exePath: installDir.resolve("ServerListExplorer.exe"),
            platform: "windows"
        };
    }

    const baseDir = isMac()
        ? Paths.get(userHome, "Library", "Application Support", "ServerListExplorer")
        : Paths.get(userHome, ".local", "share", "ServerListExplorer");

    return {
        installDir: baseDir,
        jarPath: baseDir.resolve("ServerListExplorer.jar"),
        platform: isMac() ? "mac" : "linux"
    };
};

const getLatestTag = () => {
    const json = http.getText(
        "https://api.github.com/repos/SpoilerRules/server-list-explorer/releases/latest"
    );
    const match = /"tag_name"\s*:\s*"([^"]+)"/.exec(json);
    if (!match) throw new Error("Could not parse latest release tag.");
    return match[1];
};

const copyStreamToFile = (inputStream, targetPath) => {
    Files.createDirectories(targetPath.getParent());
    const outFile = targetPath.toFile();
    const sink = Okio.buffer(Okio.sink(outFile));
    try {
        const src = Okio.buffer(Okio.source(inputStream));
        try {
            sink.writeAll(src);
            sink.flush();
        } finally {
            try {
                src.close();
            } catch (ignored) {
                chat("§eWarning: Could not close input stream, expect possible resource leak.");
            }
        }
    } finally {
        try {
            sink.close();
        } catch (ignored) {
            chat("§eWarning: Could not close output sink, expect possible resource leak.");
        }
    }
};

const http = (() => {
    const client = new OkHttpClient.Builder()
        .connectTimeout(15, TimeUnit.SECONDS)
        .readTimeout(300, TimeUnit.SECONDS)
        .build();

    const getText = (url) => {
        const req = new Request.Builder()
            .url(url)
            .build();
        const resp = client.newCall(req).execute();
        try {
            if (!resp.isSuccessful()) throw new Error(`HTTP ${resp.code()}`);
            return resp.body().string();
        } finally {
            try {
                resp.close();
            } catch (ignored) {
                chat("§eWarning: Could not close HTTP response, expect possible resource leak.");
            }
        }
    };

    const download = (url, targetPath) => {
        const req = new Request.Builder()
            .url(url)
            .build();
        const resp = client.newCall(req).execute();
        try {
            if (!resp.isSuccessful()) throw new Error(`HTTP ${resp.code()}`);
            copyStreamToFile(resp.body().byteStream(), targetPath);
        } finally {
            try {
                resp.close();
            } catch (ignored) {
                chat("§eWarning: Could not close HTTP response, expect possible resource leak.");
            }
        }
    };

    return {getText, download};
})();

const deleteDirectoryHard = (root) => {
    if (!Files.exists(root)) return {ok: true, removed: false, failures: []};
    const failures = [];
    const Visitor = Java.extend(SimpleFileVisitor, {
        visitFile(file, attrs) {
            try {
                if (isWindows()) {
                    try {
                        const view = Files.getFileAttributeView(file, DosFileAttributeView);
                        if (view) view.setReadOnly(false);
                    } catch (ignored) {
                        chat("§eWarning: Could not clear read-only flag, file may not delete.");
                    }
                    try {
                        file.toFile().setWritable(true, false);
                    } catch (ignored) {
                        chat("§eWarning: Could not set file writable, file may not delete.");
                    }
                }
                Files.deleteIfExists(file);
            } catch (e) {
                failures.push(`${file.toString()} could not be deleted because: ${String(e)}`);
            }
            return FileVisitResult.CONTINUE;
        },
        postVisitDirectory(dir, exc) {
            try {
                Files.deleteIfExists(dir);
            } catch (e) {
                failures.push(`${dir.toString()} could not be removed because: ${String(e)}`);
            }
            return FileVisitResult.CONTINUE;
        }
    });
    Files.walkFileTree(root, new Visitor());
    const exists = Files.exists(root);
    return {ok: !exists && failures.length === 0, removed: !exists, failures};
};

const moveChildrenUp = (childDir, destDir) => {
    if (!Files.exists(childDir) || !Files.isDirectory(childDir)) return;

    let dirStream;
    try {
        dirStream = Files.list(childDir);
        const list = dirStream.toArray();

        for (const p of list) {
            const target = destDir.resolve(p.getFileName().toString());
            try {
                if (Files.isDirectory(p)) {
                    Files.createDirectories(target);
                    moveChildrenUp(p, target);
                } else {
                    Files.createDirectories(target.getParent());
                    Files.move(p, target, StandardCopyOption.REPLACE_EXISTING);
                }
            } catch (e) {
                chat(`§eWarning: Could not move ${p.toString()} to ${target.toString()} because: ${String(e)}`);
            }
        }
    } catch (e) {
        chat(`§cError listing directory ${childDir.toString()}: ${String(e)}`);
    } finally {
        if (dirStream) {
            try {
                dirStream.close();
            } catch (ignored) {
                chat(`§eWarning: Could not close directory stream for ${childDir.toString()}, expect possible resource leak.`);
            }
        }
    }
};

const ensureParentDirs = (path) => {
    const parent = path.getParent();
    if (parent == null) return;

    try {
        Files.createDirectories(parent);
    } catch (e) {
        // if something exists here and is not a directory, replace it with a directory
        if (Files.exists(parent) && !Files.isDirectory(parent)) {
            try {
                Files.delete(parent);
            } catch (ignored) {
                chat(`§eWarning: Could not delete non-directory at ${parent.toString()}, expect possible issues.`);
            }
            try {
                Files.createDirectories(parent);
            } catch (inner) {
                chat(`§cFailed to create directory at ${parent.toString()} because: ${String(inner)}`);
                throw inner;
            }
        } else {
            chat(`§cFailed to create parent directories for ${path.toString()} because: ${String(e)}`);
            throw e;
        }
    }
};

const extractAndFlattenZip = (zipPath, destDir) => {
    Files.createDirectories(destDir);
    const destNorm = destDir.toAbsolutePath().normalize();

    const zip = new ZipFile(zipPath.toFile());
    try {
        const entries = zip.entries();
        while (entries.hasMoreElements()) {
            const entry = entries.nextElement();
            let name = String(entry.getName()).replace(/\\/g, "/");

            // flatten top-level folder
            if (name.startsWith("ServerListExplorer/")) {
                name = name.substring("ServerListExplorer/".length);
            }
            if (name.length === 0) continue;

            const outPath = destDir.resolve(name).toAbsolutePath().normalize();

            // zip slip guard
            if (!outPath.startsWith(destNorm)) {
                chat(`§eSkipping suspicious entry: §f${name}`);
                continue;
            }

            // directory entry (or name ends with '/')
            if (entry.isDirectory() || name.endsWith("/")) {
                try {
                    Files.createDirectories(outPath);
                } catch (e) {
                    chat(`§cFailed to create directory ${outPath.toString()} because: ${String(e)}`);
                }
                continue;
            }

            ensureParentDirs(outPath);

            // write file using okio
            const inStream = zip.getInputStream(entry);
            try {
                copyStreamToFile(inStream, outPath);
            } catch (e) {
                chat(`§cFailed to extract ${name} to ${outPath.toString()} because: ${String(e)}`);
            } finally {
                try {
                    inStream.close();
                } catch (ignored) {
                    chat(`§eWarning: Could not close input stream for ${name}, expect possible resource leak.`);
                }
            }
        }
    } finally {
        try {
            zip.close();
        } catch (ignored) {
            chat("§eWarning: Could not close ZIP file, expect possible resource leak.");
        }
    }
};

script.registerCommand({
    name: "sle",
    aliases: ["serverlistexplorer"],
    hub: true,
    subcommands: [
        {
            name: "install",
            onExecute() {
                chat("§7Starting install...");
                runAsync("sle-install", () => {
                    try {
                        chat("§7Fetching latest release tag...");
                        const tag = getLatestTag();
                        const version = tag.replace(/^v/, "");

                        const paths = getPaths();
                        const tmpDir = Paths.get(SystemJ.getProperty("java.io.tmpdir"));
                        Files.createDirectories(tmpDir);

                        const withPrefix = new MessageMetadata(true, null, true, 1);

                        if (paths.platform === "windows") {
                            const zipName = `ServerListExplorer-minified_windows-x86_64-portable.zip`;
                            const zipUrl = `https://github.com/SpoilerRules/server-list-explorer/releases/download/${tag}/${zipName}`;
                            const tmpZip = tmpDir.resolve(zipName);

                            chat(`§7Downloading §a${zipName}...`);

                            http.download(zipUrl, tmpZip);
                            chat("§aDownload complete. Extracting...");

                            const destDir = paths.installDir;
                            deleteDirectoryHard(destDir);
                            Files.createDirectories(destDir);

                            extractAndFlattenZip(tmpZip, destDir);

                            const nested = destDir.resolve("ServerListExplorer");
                            if (Files.exists(nested) && Files.isDirectory(nested)) {
                                moveChildrenUp(nested, destDir);
                                deleteDirectoryHard(nested);
                            }

                            Files.deleteIfExists(tmpZip);
                            const line = ClientChat.regular("§aInstalled to §f")
                                .append(ClientChat.clickablePath(new File(String(destDir))));
                            ClientChat.chat(line, withPrefix);
                        } else {
                            const jarFileName = "ServerListExplorer-all-minified.jar";
                            const jarUrl = `https://github.com/SpoilerRules/server-list-explorer/releases/download/${tag}/${jarFileName}`;
                            const tmpJar = tmpDir.resolve(jarFileName);

                            chat(`§7Downloading §a${jarFileName}...`);

                            http.download(jarUrl, tmpJar);
                            chat("§aDownload complete. Installing...");

                            const destDir = paths.installDir;
                            deleteDirectoryHard(destDir);
                            Files.createDirectories(destDir);

                            const finalJar = paths.jarPath;
                            Files.move(tmpJar, finalJar, StandardCopyOption.REPLACE_EXISTING);

                            const line = ClientChat.regular("§b[§3SLE§b] §aInstalled to §f")
                                .append(ClientChat.clickablePath(new File(String(finalJar))));
                            ClientChat.chat(line, withPrefix);
                        }
                        chat("§aServer List Explorer is ready. Use §f.sle run §ato start Server List Explorer.");
                    } catch (e) {
                        chat(`§cServer List Explorer installation failed: §f${String(e)}`);
                    }
                });
            }
        },
        {
            name: "uninstall",
            onExecute() {
                runAsync("sle-uninstall", function () {
                    try {
                        const installDir = getPaths().installDir;

                        if (!Files.exists(installDir)) {
                            chat("§eServer List Explorer is not installed.");
                            return;
                        }

                        chat("§7Uninstalling Server List Explorer...");
                        const res = deleteDirectoryHard(installDir);

                        if (res.ok || !Files.exists(installDir)) {
                            chat("§aServer List Explorer has been uninstalled.");
                        } else {
                            chat("§6Server List Explorer was partially uninstalled. Some items could not be removed (they may be in use).");
                            chat(`§6Delete manually if needed: §f${installDir.toString()}`);
                        }
                    } catch (e) {
                        try {
                            const withPrefix = new MessageMetadata(false, null, true, 1);

                            const line = ClientChat.regular("§b[§3SLE§b] §cServer List Explorer uninstall failed: §f")
                                .append(ClientChat.clickablePath(new File(String(installDir))))
                                .append(ClientChat.regular(` §7(§cError message: §f${String(e)}§7)`));

                            ClientChat.chat(line, withPrefix);
                        } catch (inner) {
                            chat(`§cServer List Explorer uninstall failed: §f${String(e)}`);
                        }
                    }
                });
            }
        },
        {
            name: "run",
            onExecute() {
                runAsync("sle-run", function () {
                    chat("§7Launching Server List Explorer...");
                    try {
                        const paths = getPaths();
                        if (paths.platform === "windows") {
                            if (!Files.exists(paths.exePath)) {
                                chat("§cServer List Explorer is not installed. Use §f.sle install §cfirst.");
                                return;
                            }
                            runDetached([paths.exePath.toString()], paths.installDir);
                            chat("§aLaunched Server List Explorer.");
                        } else {
                            const jarPath = paths.jarPath;
                            if (!Files.exists(jarPath)) {
                                chat("§cNot installed. Use §f.sle install §cfirst.");
                                return;
                            }
                            runDetached([javaBin(), "-jar", jarPath.toString()], paths.installDir);
                            chat("§aLaunched Server List Explorer (JAR version).");
                        }
                    } catch (e) {
                        chat(`§cRun failed: §f${String(e)}`);
                    }
                });
            }
        },
        {
            name: "path",
            onExecute() {
                try {
                    const installDir = getPaths().installDir;

                    if (!Files.exists(installDir)) {
                        chat("§eServer List Explorer is not installed.");
                        return;
                    }

                    const withPrefix = new MessageMetadata(true, null, true, 1);

                    const line = ClientChat.regular("§b[§3SLE§b] §7Location: §f")
                        .append(ClientChat.clickablePath(new File(String(installDir))));

                    ClientChat.chat(line, withPrefix);
                } catch (e) {
                    chat(`§cPath lookup failed: §f${String(e)}`);
                }
            }
        },
    ]
});
